/* $Id: stats.c,v 1.7 1999/03/29 07:37:10 roca Exp $ */
/*
 * stats.c
 *
 *	Modified by : V.Roca
 *	Date	    : dec 1997
 *
 *	bench/benchd statistic routines.
 */
/*
 * Copyright (c) 1998/1999 - V.Roca (vincent.roca@lip6.fr) All rights reserved
 * This tool is provided as is, without any warranty.
 * Permission to use, copy and modify is provided for non commercial
 * purposes as long as this notice appears on all copies.
 */

#include "bench.h"
#include "bench_proto.h"


/****** session start/end ******/

/*
 * Print the starting and end times of each test.
 * Can be used for synchronizing the various measurements collected
 * on a given machine, and create a single plot.
 */
void
PrintStartTime (int	conn)
{
	struct timeval	start;

	gettimeofday(&start, NULL);
#ifdef SERVER
	PRINT_OUT((stdout, "SERVER %s conn %d\tstart_time=%ld.%06ld s\n",
		provider, conn, start.tv_sec, start.tv_usec))
#else
	PRINT_OUT((stdout, "CLIENT %s conn %d\tstart_time=%ld.%06ld s\n",
		provider, conn, start.tv_sec, start.tv_usec))
#endif
}


void
PrintEndTime (int	conn)
{
	struct timeval	end;

	gettimeofday(&end, NULL);
#ifdef SERVER
	PRINT_OUT((stdout, "SERVER %s conn %d\tend_time=%ld.%06ld s\n",
		provider, conn, end.tv_sec, end.tv_usec))
#else
	PRINT_OUT((stdout, "CLIENT %s conn %d\tend_time=%ld.%06ld s\n",
		provider, conn, end.tv_sec, end.tv_usec))
#endif
}

/****** statistic gathering routines ******/


#ifdef SERVER
/*
 *	Used by both the IntUser1 and (Sock)RecvUDPMsg functions
 */
void
ResetStats (void)
{
	totdata = 0;
	totmsg = 0;
	tottt = 0.0;		/* 17/3/99: added */
	recdata = 0;
	recmsg = 0;
	rectt = 0.0;
	min_recdata = 1000000;
	max_recdata = 0;
	min_tt = 1000000.0;
	max_tt = 0.0;
	fthrumin = 1000000.0;
	fthrumax = 0.0;
	udp_loss = 0;
	udp_count = 0;
	totudploss = 0;
	gettimeofday(&time0, NULL);
	time1 = time0;
}
#endif /* SERVER */


/*
 *	Used by both server and client to register/print partial
 *	statistics periodically.
 */
void
Stats (int	conn,
       int	data,
       int	msg,
       float	tt)
{
	float	fthru, fmsg, dt;
	float	fdelay;		/* latency */
#ifdef SERVER
	float	av_recdata;	/* average recv buffer length */
	char	add_stats[256];

	if (mode == RTT_MODE)
		return;
	if (raw) {
		/*
		 * do not calculate stats but print raw info
		 */
		gettimeofday(&time2, NULL);
		if (udp) {
			PRINT_OUT((stdout,
				"%ld.%06ld\tsz=%d seq=%d TT=%.3f L=%d\n",
				time2.tv_sec, time2.tv_usec,
				data, udp_count - 1, tt * 1000.0, udp_loss))
			udp_loss = 0;
		} else {
			PRINT_OUT((stdout,
				"%ld.%06ld\tsz=%d\n",
				time2.tv_sec, time2.tv_usec, data))
		}
		return;
	}
#else  /* CLIENTS case */
	if (mode == RTT_MODE && raw) {
		/*
		 * do not calculate stats but print raw info
		 */
		gettimeofday(&time2, NULL);
		if (udp) {
			/* XXX: very few stats available yet! */
			PRINT_OUT((stdout,
				"%ld.%06ld\tsz=%d seq=%d TT=%.3f\n",
				time2.tv_sec, time2.tv_usec,
				data, udp_count - 1, tt * 1000.0))
		} else {
			PRINT_OUT((stdout,
				"%ld.%06ld\tsz=%d TT=%.3f\n",
				time2.tv_sec, time2.tv_usec, data, tt * 1000.0))
		}
		return;
	}

#endif /* SERVER */
	recdata += data;
	recmsg += msg;
#ifdef NEVERDEF
	printf("Stats: tt=%f, rectt=%f\n", tt, rectt);
#endif /* NEVERDEF */
#ifdef SERVER
	/*
	 * Additional statistics on receive buffer filling
	 */
	if (data > max_recdata)
		max_recdata = data;
	if (data < min_recdata)
		min_recdata = data;
	/*
	 * ... and transit time 
	 */
	rectt += tt;
	if (mode == TT_MODE) {
		if (tt > max_tt)
			max_tt = tt;
		if (tt < min_tt)
			min_tt = tt;
	}
	gettimeofday(&time2, NULL);
#else  /* CLIENT case */
	/* gettimeofday of time2 done in SendMsg */
#endif
 	if (time2.tv_sec >= time1.tv_sec + tmsg) {
		dt = time2.tv_sec - time1.tv_sec +
		     (time2.tv_usec - time1.tv_usec) * 1e-6;
		/*
		 * VR 17/3/98: done here to avoid differences in sender
		 * and receiver measures. Of course, printf execution time
		 * is included in measures  (if it is a problem, use
		 * large trace periods) !
		 */
		gettimeofday(&time1, NULL);

		fthru = recdata / (dt * 1024.0);
		if (fthru < fthrumin)
			fthrumin = fthru;
		if (fthru > fthrumax)
			fthrumax = fthru;
		fmsg = recmsg / dt;
		if (fmsg > 0.0)
			fdelay = 1000.0 / fmsg;
		else
			fdelay = -1.0;
#ifdef SERVER
		if (recmsg > 0)
			av_recdata = recdata / recmsg;
		else
			av_recdata = -1;
		if (verbose) {
			/* additional stats */
			sprintf(add_stats, " S=%d/%.1f/%d bytes/msg D=%.3f ms",
				min_recdata, av_recdata, max_recdata, fdelay);
		} else {
			add_stats[0] = '\0';
		}
		if (udp) {
			if (mode == TT_MODE) {
				PRINT_OUT((stdout,
				"SERVER %s conn %d RCV\tTT=%.3f ms M=%.1f msg/s L=%.1f lost_msg/s %s\n",
				provider, conn, 1000.0 * rectt / (float)recmsg,
				fmsg, (float)udp_loss / dt, add_stats))
				tottt += rectt;
				rectt = 0.0;
			} else {
				PRINT_OUT((stdout,
				"SERVER %s conn %d RCV\tT=%s M=%.1f msg/s L=%.1f lost_msg/s %s\n",
				provider, conn,
				unit_transl(fthru, TRANSL_WITH_UNITS),
				fmsg, (float)udp_loss / dt, add_stats))
			}
			totudploss += udp_loss;
			udp_loss = 0;
		} else if (tcp) {
			PRINT_OUT((stdout,
			"SERVER %s conn %d RCV\tT=%s M=%.1f msg/s %s\n",
			provider, conn, unit_transl(fthru, TRANSL_WITH_UNITS),
			fmsg, add_stats))
		}
		min_recdata = 1000000;
		max_recdata = 0;
#else  /* CLIENT case */
		PRINT_OUT((stdout,
			"CLIENT %s conn %d %s\tT=%s M=%.1f msg/s D=%.3f ms\n",
			provider, conn, (mode == RTT_MODE ? "SNDRCV" : "SND"),
			unit_transl(fthru, TRANSL_WITH_UNITS), fmsg, fdelay))
#endif
		totdata += recdata;
		totmsg += recmsg;
		recdata = recmsg = 0.0;
#ifdef NEVERDEF
		/*
		 * VR 22/07/94: for more precision, don't include PRINT_OUT
		 * in the estimation.
		 * VR 17/3/98: FALSE, leads to differences between sender
		 * and receiver measures !!! Moved above.
		 */
		gettimeofday(&time1, NULL);
#endif /* NEVERDEF */
	}
}


/*
 *	Used by both server and client to print statistics at
 *	the end of the test.
 */
void
EndStats (void)
{
	float	fthru, fmsg, dt;
	float	fdelay;		/* latency */
	char	stats[1024];
	char	add_stats[256];
	int	off = 0;	/* offset for copy to the stats buf */
#ifdef SERVER
	float	av_totdata;	/* average recv buffer length */
	float	av_tottt;	/* average transit time */

	if (mode == RTT_MODE)
		return;
	if (raw)
		return;
#else  /* CLIENT case */
	if (dtonly) {
		/*
		 * Do statistics only during data transfer.
		 */
		gettimeofday(&time2, NULL);
	}
	if (sock)
		close(conn_fd);
#ifdef XTI
	else if (xti) {
		if (!udp) {
			RelCon(conn_fd, &xti_info);
			t_close(conn_fd);
		}
	}
#endif /* XTI */
#endif
	if (!dtonly) {
		/*
		 * Include connection release.
		 */
		gettimeofday(&time2, NULL);
	}
	dt = time2.tv_sec - time0.tv_sec +
	     (time2.tv_usec - time0.tv_usec) * 1e-6;
	totdata += recdata;
	totmsg += recmsg;
	fthru = totdata / (dt * 1024.0);
	if (fthru < fthrumin)
		fthrumin = fthru;
	if (fthru > fthrumax)
		fthrumax = fthru;
	fmsg = totmsg / dt;
	if (fmsg > 0.0)
		fdelay = 1000.0 / fmsg;
	else
		fdelay = -1.0;

#ifdef SERVER /* { */

	tottt += rectt;
	if (totmsg > 0) {
		av_totdata = totdata / totmsg;
		av_tottt = 1000.0 * tottt / (float)totmsg;
	} else {
		av_totdata = -1;
		av_tottt = -1;
	}

	if (udp) {
		totudploss += udp_loss;		/* lost pkts of last period */
		if (verbose) {
			/* additional stats */
			sprintf(add_stats,
				"total_recv=%d bytes in %d msg total_lost=%d msg test_dur=%.3f s averS=%.1f bytes/msg averD=%.3f ms",
				totdata, totmsg, totudploss, dt, av_totdata,
				fdelay);
		} else {
			add_stats[0] = '\0';
		}

		if (mode == TT_MODE) {
			PRINT_OUT((stdout,
				"SERVER_SUMMARY %s conn %d RCV\taverTT=%.3f ms (minTT/maxTT=%.3f/%.3f ms) averM=%.1f msg/s averL=%.1f lost_msg/s %s\n",
				provider, connex,
				av_tottt, min_tt * 1000.0, max_tt * 1000.0,
				fmsg, (float)totudploss / dt,
				add_stats))
			tottt += rectt;
			rectt = 0.0;
		} else {
			/*
			 * throughput and RTT modes
			 */
			off = sprintf(stats, "SERVER_SUMMARY %s conn %d RCV\taverT=%s",
				provider, connex,
				unit_transl(fthru, TRANSL_WITH_UNITS));
			off += sprintf(stats + off, " (minT/maxT=%s",
				unit_transl(fthrumin, TRANSL_WO_UNITS));
			off += sprintf(stats + off, "/%s",
				unit_transl(fthrumax, TRANSL_WO_UNITS));
			sprintf(stats + off, " %s) averM=%.1f msg/s averL=%.1f lost_msg/s %s\n",
				unit_transl(0, TRANSL_ONLY_UNITS),
				fmsg, (float)totudploss / dt, add_stats);
			totudploss = 0;
			udp_loss = 0;
		}

	} else if (tcp) {
		if (verbose) {
			/* additional stats */
			sprintf(add_stats,
				"total_recv=%d bytes in %d msg test_dur=%.3f s averS=%.1f bytes/msg averD=%.3f ms",
				totdata, totmsg, dt, av_totdata, fdelay);
		} else {
			add_stats[0] = '\0';
		}

		off = sprintf(stats, "SERVER_SUMMARY %s conn %d RCV\taverT=%s",
			provider, connex,
			unit_transl(fthru, TRANSL_WITH_UNITS));
		off += sprintf(stats + off, " (minT/maxT=%s",
			unit_transl(fthrumin, TRANSL_WO_UNITS));

		off += sprintf(stats + off, "/%s",
			unit_transl(fthrumax, TRANSL_WO_UNITS));
		sprintf(stats + off, " %s) averM=%.1f msg/s %s\n",
			unit_transl(0, TRANSL_ONLY_UNITS), fmsg, add_stats);
	}

#else  /* } CLIENT case { */

	if (verbose) {
		/* additional stats */
		sprintf(add_stats, "total_snt=%d bytes in %d msg test_dur=%.3f s",
			totdata, totmsg, dt);
	} else {
		add_stats[0] = '\0';
	}

	off = sprintf(stats, "CLIENT_SUMMARY %s conn %d %s\taverT=%s",
		provider, connex, (mode == RTT_MODE ? "SNDRCV" : "SND"),
		unit_transl(fthru, TRANSL_WITH_UNITS));
	off += sprintf(stats + off, " (minT/maxT=%s", unit_transl(fthrumin, TRANSL_WO_UNITS));
	off += sprintf(stats + off, "/%s", unit_transl(fthrumax, TRANSL_WO_UNITS));
	sprintf(stats + off, " %s) averM=%.1f msg/s averD=%.3f ms %s\n",
		unit_transl(0, TRANSL_ONLY_UNITS),
		fmsg, fdelay, add_stats);

#endif  /* } SERVER/CLIENT */
	PRINT_OUT((stdout, "%s", stats))
}


/*
 * unit_transl
 *
 * performs unit translation and returns the approapriate string
 * controlled by the following format parameter:
 *	TRANSL_WITH_UNITS	"value units"
 *	TRANSL_WO_UNITS		"value"
 *	TRANSL_ONLY_UNITS	"units"
 * input value is in kilobytes (1024)
 *
 * WARNING: buffer returned must be used before calling this function again !
 */
char *
unit_transl (float	value,
	     int	format)
{
	static char	buf[128];	/* static since returned to caller! */
	char		valstring[128];
	char		*precision = NULL;	/* init to avoid warning */
	char		*unitstring = NULL;	/* init to avoid warning */

	switch (units) {
	case MEGABITS_PER_SEC:
		value *= 8.0 * 1024.0 * 1.0e-6;
		precision = "%.3f";
		unitstring = "Mbps";
		break;
	case KILOBITS_PER_SEC:
		value *= 8.0 * 1024.0 * 1.0e-3;
		precision = "%.3f";
		unitstring = "kbps";
		break;
	case KILOBYTES_PER_SEC:
		/* value unchanged */
		precision = "%.3f";
		unitstring = "kBytes/s";
		break;
	}
	switch (format) {
	case TRANSL_WITH_UNITS:
		sprintf(valstring, precision, value);
		sprintf(buf, "%s %s", valstring, unitstring);
		break;
	case TRANSL_WO_UNITS:
		sprintf(buf, precision, value);
		break;
	case TRANSL_ONLY_UNITS:
		sprintf(buf, "%s", unitstring);
		break;
	}
	return buf;
}
