/* $Id: bench.c,v 1.6 1999/05/11 11:59:25 roca Exp $ */
/*
 * bench.c
 *
 *	Modified by : V.Roca
 *	Date	    : dec 1997
 *
 *	Client part of the performance evaluation tool.
 *
 */
/*
 * Copyright (c) 1998/1999 - V.Roca (vincent.roca@lip6.fr) All rights reserved
 * This tool is provided as is, without any warranty.
 * Permission to use, copy and modify is provided for non commercial
 * purposes as long as this notice appears on all copies.
 */

#include "bench.h"
#include "bench_proto.h"


static pid_t tab_pid[MAX_CONNECT];
static int sync_fd;			/* fd to the synchronization conn */
static int flag_on;			/* indicates SIGUSR1 occured */


void
main (int	argc,
      char	*argv[])
{
	ParseCommandLine (argc, argv);
	PrintParams ();
	StartClient ();
	exit (0);
}


void StartClient (void)
{
	int status;
	int numcon;

	if (udp) {		/* includes the bcast and mcast cases */
		connex = 0;
		Client(connex);
	} else if (tcp) {
		signal(SIGCHLD, SIG_IGN);
		/*
		 * Fork for each connection.
		 */
		for (connex = 1 ; connex <= nbconn ; connex++)
			Client(connex);
		/*
		 * Send a signal to each child to ask him to create
		 * the transport endpoint and open the connection.
		 * Delay each opening if asked.
		 */
		for (numcon = 0 ; numcon < nbconn ; numcon++) {
			flag_on = 0;
			kill (tab_pid[numcon], SIGUSR1);
			while (!flag_on) {
				sleep (1);
			}
			if (delay) {
				/* sleep(delay); */
				usleep(delay*1000); /* delay is in millisec */
			}
		}

		/*
		 * Start synchro with benchd if asked.
		 */
		if (syncf)
			sync_start();

		/*
		 * Send a signal to each child to trigger the
		 * data transfer.
		 */
		for (numcon = 0 ;numcon < nbconn ; numcon++)
			kill (tab_pid[numcon], SIGUSR1);

		/*
		 * Wait ... untill the father receives a SIGUSR2
		 * from a child (abnormal termination) or untill all
		 * the childs terminate.
		 */
		for (numcon = 0 ;numcon < nbconn ; numcon++)
			wait(&status);

		if (syncf)	/* stop synchro with benchd */
			sync_stop();
	}
}


void
Client (int	num)
{
	if (udp || bcast) {
#if defined(LINUX) || defined(SOLARIS)
		if (pin > 0) {
			/* pin all current and future segs to memory */
			if (mlockall(MCL_CURRENT | MCL_FUTURE) < 0) {
				perror("bench: mlockall");
				EXIT(num, -1);
			}
		}
#endif /* LINUX || SOLARIS */
		if (sock)
			SockOpen(0);
#ifdef XTI
		else if (xti)
			XTIOpen(0);
#endif /* XTI */
		/*
		 * Run the CPU usage and lockstat tools.
		 */
		if (cpustat) {
			if (cpustat == 1)
				system("sar -uw 5 2 &");
			else if (cpustat == 3)
				system("vmstat 2 6 &");
#if defined(CPUSTAT) && defined(BOSX41)
			else
				initstat();
#endif /* CPUSTAT && BOSX41 */
		}
#ifdef BOSX41
		if (lockstat)
			system("lockstat -a -t200 15 1 &");
#endif /* BOSX41 */
		if (syncf)	/* start synchro with benchd */
			sync_start();

		if (sock)
			SockSendMsg(0);
#ifdef XTI
		else if (xti)
			XTISendMsg(0);
#endif /* XTI */

		if (syncf)	/* stop synchro with benchd */
			sync_stop();
#if defined(CPUSTAT) && defined(BOSX41)
		if (cpustat == 2 && !access(TMPFILE, F_OK))
			stopstat();	/* must be stopped explicitely */
#endif /* CPUSTAT && BOSX41 */

	} else {
		signal(SIGUSR1, (sighandler_t)IntUser1);
		flag_on = 0;

		switch (tab_pid[num-1] = fork()) {

		case -1: 	/* Error */
			perror ("bench: fork failed");
			exit(1);
			break;

		case 0:		/* Child */
#ifdef LINUX
			if (pin > 0) {
				/* pin all current and future segs to memory */
				if (mlockall(MCL_CURRENT | MCL_FUTURE) < 0) {
					perror("bench: mlockall");
					EXIT(1, -1);
				}
			}
#endif /* LINUX */
			/*
			 * Wait signal from parent process, then
			 * open a transport endpoint and open the
			 * connection.
			 */
			while (!flag_on)
				sleep(2);
			if (sock)
				SockOpen(num);
#ifdef XTI
			else if (xti)
				XTIOpen(num);
#endif /* XTI */
			/*
			 * Tell the parent process the opening
			 * is over.
			 * Wait for the following signal, then
			 * send data.
			 */
			flag_on = 0;
			kill (getppid(), SIGUSR1);	
			while (!flag_on)
				sleep (1);
			/*
			 * Run the CPU usage and lockstat tools.
			 * Done only by the first client.
			 */
			if (cpustat && num == 1) {
				if (cpustat == 1)
					system("sar -uw 5 2 &");
				else if (cpustat == 3)
					system("vmstat 2 6 &");
#if defined(CPUSTAT) && defined(BOSX41)
				else
					initstat();
#endif /* CPUSTAT && BOSX41 */
			}
#ifdef BOSX41
			if (lockstat && num == 1)
				system("lockstat -a -t200 15 1 &");
#endif /* BOSX41 */
			if (sock)
				SockSendMsg(num);
#ifdef XTI
			else if (xti)
				XtiSendMsg(num);
#endif /* XTI */

#if defined(CPUSTAT) && defined(BOSX41)
			/*
			 * The first child to close its connection
			 * also triggers statistic printing.
			 */
			if (cpustat == 2 && !access(TMPFILE, F_OK))
				stopstat();	/* must be stopped explicitely*/
#endif /* CPUSTAT && BOSX41 */
			exit(0);
			break;

		default:	/* Father */
			signal(SIGUSR2, (sighandler_t)IntUser2);
			break;
		}
	}
}


/****** Signal processing, and bench/benchd synchronization ******/

/*
 * Triggers the next processing step for the parent and/or the child process.
 */
void
IntUser1 (void)
{
	flag_on = 1;
	signal(SIGUSR1, (sighandler_t)IntUser1);
}

/*
 * Performed by the parent process in case of an abnormal exit (cf EXIT()).
 */
void
IntUser2 (void)
{
	exit(1);
}

/*
 * We received a SIGINT.
 */
void
Interrupted (void)
{
	EndStats();
	PrintEndTime(connex);
	exit(0);
}


/*
 * Start synchronization.
 * We create an additional TCP connection where we give some
 * information to the remote benchd at test start.
 */
void
sync_start (void)
{
	char			syncbuf[256];
	int			syncbuf_len;
	int			len;
	struct hostent		*h;
	struct servent		*sp;

	/*
	 * Information sent to peer.
	 */
	sprintf(syncbuf, SYNC_FORMAT, lgthmsg, nbconn);
	syncbuf_len = strlen(syncbuf) + 1;	/* strlen doesn't include '\0'*/


	if ((h = gethostbyname(rname)) == NULL) {
		PRINT_OUT((stdout, "bench: don't know remotename ""%s""\n",
			rname))
		exit(-1);
	}

	if (sock) {
		struct sockaddr_in	sa;

		memset((char *)&sa, 0, sizeof(sa));
		memcpy((char *)&sa.sin_addr.s_addr, h->h_addr_list[0],
			h->h_length);
		sa.sin_family = AF_INET;
		if (eport)
			sa.sin_port = htons(atoi(myport) + 1);
		else {
			if (!(sp = getservbyname("benchd_sync", "tcp"))) {
				PRINT_OUT((stdout, "bench sync: don't know server name ""benchd_sync""\n"))
				PRINT_OUT((stdout, "bench/benchd not registered in /etc/services or -p option forgotten\n"))
				exit(-1);
			}
			sa.sin_port = htons(sp->s_port);
		}

		if ((sync_fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
			perror("bench sync: socket");
			exit(-1);
		}
		if (connect(sync_fd, (struct sockaddr*) &sa, sizeof(sa)) < 0) {
			perror("bench sync: connect");
			exit(-1);
		}
		if ((len = send(sync_fd, syncbuf, syncbuf_len, 0)) < syncbuf_len) {
			perror("bench sync: send");
			exit(-1);
		}
	}
#ifdef XTI
	else if (xti) {
		struct t_call		*sndcall,
					*rcvcall;
		struct netbuf		*addr;
		struct sockaddr_in	*sa;
		struct xtitp		sync_tp;

		if (STD_stack)
			sync_tp.tp_id = TPID_STD_TCP;
		else if (CCA_stack)
			sync_tp.tp_id = TPID_CCA_UDP;
		else {
			t_error("bench sync: bad stack");
			exit(-1);
		}
		if ((sync_fd = t_open(sync_tp.tp_name, O_RDWR, &sync_xti_info)) < 0) {
			t_error("bench sync: t_open failed");
			exit(-1);
		}
		if (((sndcall = (struct t_call *)(t_alloc(sync_fd, T_CALL_STR,
		                                         T_ALL))) == NULL) ||
		    ((rcvcall = (struct t_call *)(t_alloc(sync_fd, T_CALL_STR,
		                                         T_ALL))) == NULL)) {
			t_error("bench sync: t_alloc failed");
			exit(-1);
		}
		addr = &sndcall->addr;
		sa = (struct sockaddr_in *)addr->buf;
		sa->sin_family = AF_INET;
		memcpy((char *)&sa->sin_addr.s_addr, h->h_addr_list[0],
			h->h_length);
		sa->sin_port = atoi(myport);
		addr->len = sizeof(struct sockaddr_in);

		if (t_bind(sync_fd, NULL, NULL) < 0) {
			t_error("bench sync: t_bind failed");
			exit(-1);
		}
		if (t_connect(sync_fd, sndcall, rcvcall) < 0) {
			t_error("bench sync: t_connect failed");
			exit(-1);
		}
		if (t_free((char *)sndcall, T_CALL_STR) < 0 ||
		    t_free((char *)rcvcall, T_CALL_STR) < 0) {
			t_error("bench sync: t_free failed");
			exit(-1);
		}
		if ((len = t_snd(sync_fd, syncbuf, syncbuf_len, 0)) < syncbuf_len) {
			perror("bench sync: send");
			exit(-1);
		}
	}
#endif /* XTI */
}

/*
 * Stop synchronization.
 */
void
sync_stop (void)
{
	if (sock)
		close(sync_fd);
#ifdef XTI
	else if (xti) {
		RelCon(sync_fd, &sync_xti_info);
		t_close(sync_fd);
	}
#endif /* XTI */
}

